/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.cloudwatchlogs.emf.sinks;

import com.fasterxml.jackson.core.JsonProcessingException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.cloudwatchlogs.emf.exception.EMFClientException;
import software.amazon.cloudwatchlogs.emf.model.MetricsContext;
import software.amazon.cloudwatchlogs.emf.sinks.Endpoint;
import software.amazon.cloudwatchlogs.emf.sinks.ISink;
import software.amazon.cloudwatchlogs.emf.sinks.SocketClient;
import software.amazon.cloudwatchlogs.emf.sinks.SocketClientFactory;
import software.amazon.cloudwatchlogs.emf.sinks.retry.RetryStrategy;
import software.amazon.cloudwatchlogs.emf.util.StringUtils;

public class AgentSink
implements ISink {
    private static final Logger log = LoggerFactory.getLogger(AgentSink.class);
    private final String logGroupName;
    private final String logStreamName;
    private final SocketClient client;
    private final ExecutorService executor;
    private final Supplier<RetryStrategy> retryStrategyFactory;
    private final LinkedBlockingQueue<Runnable> queue;

    public AgentSink(String logGroupName, String logStreamName, Endpoint endpoint, SocketClientFactory clientFactory, int asyncQueueDepth, Supplier<RetryStrategy> retryStrategy) {
        this.logGroupName = logGroupName;
        this.logStreamName = logStreamName;
        this.client = clientFactory.getClient(endpoint);
        this.queue = new LinkedBlockingQueue(asyncQueueDepth);
        this.executor = this.createSingleThreadedExecutor();
        this.retryStrategyFactory = retryStrategy;
    }

    private ExecutorService createSingleThreadedExecutor() {
        return new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, this.queue, new ThreadPoolExecutor.DiscardOldestPolicy());
    }

    @Override
    public void accept(MetricsContext context) {
        if (this.executor.isShutdown()) {
            throw new EMFClientException("Attempted to write data to a sink that has been previously shutdown.");
        }
        if (!StringUtils.isNullOrEmpty(this.logGroupName)) {
            context.putMetadata("LogGroupName", this.logGroupName);
        }
        if (!StringUtils.isNullOrEmpty(this.logStreamName)) {
            context.putMetadata("LogStreamName", this.logStreamName);
        }
        try {
            for (String event : context.serialize()) {
                this.executor.submit(new Sender(event, this.client, this.retryStrategyFactory));
            }
        }
        catch (JsonProcessingException e) {
            log.error("Failed to serialize the metrics with the exception: ", (Throwable)e);
        }
    }

    @Override
    public CompletableFuture<Void> shutdown() {
        this.executor.shutdown();
        return CompletableFuture.supplyAsync(() -> {
            try {
                while (!this.executor.awaitTermination(1000L, TimeUnit.MILLISECONDS)) {
                    log.debug("Waiting for graceful shutdown to complete. {} tasks pending.", (Object)(this.queue.size() + 1));
                }
            }
            catch (InterruptedException e) {
                log.warn("Thread terminated while awaiting shutdown.");
            }
            return null;
        });
    }

    private static class Sender
    implements Runnable {
        private final String event;
        private final SocketClient client;
        private final Supplier<RetryStrategy> retryStrategyFactory;

        @Override
        public void run() {
            if (!StringUtils.isNullOrEmpty(this.event)) {
                try {
                    this.sendMessageForMaxAttempts();
                }
                catch (InterruptedException e) {
                    log.warn("Thread was interrupted while sending EMF event.");
                }
            }
        }

        private void sendMessageForMaxAttempts() throws InterruptedException {
            RetryStrategy backoff = null;
            for (int i = 0; i < 100; ++i) {
                try {
                    this.client.sendMessage(this.event + "\n");
                    return;
                }
                catch (Exception e) {
                    log.debug("Failed to write the message to the socket. Backing off and trying again.", (Throwable)e);
                    backoff = backoff != null ? backoff : this.retryStrategyFactory.get();
                    Thread.sleep(backoff.next());
                    continue;
                }
            }
        }

        public Sender(String event, SocketClient client, Supplier<RetryStrategy> retryStrategyFactory) {
            this.event = event;
            this.client = client;
            this.retryStrategyFactory = retryStrategyFactory;
        }
    }
}

